#include <iostream>
#include <vector>
#include <thread>
#include <mutex>
#include <functional>
#include <queue>
#include <future>
#include <condition_variable>

class ThreadPool {
public:
    ThreadPool(size_t num_threads) : _stop(false) {
        for(size_t i = 0; i < num_threads; ++i) {
            _workers.emplace_back([this] {
                while(true) {
                    std::function<void()> task;
                    {
                        std::unique_lock<std::mutex> lock(_queue_mutex);
                        _condition.wait(lock, [this]{ return this->_stop || !this->_tasks.empty(); });
                        if(this->_stop && this->_tasks.empty())
                            return;
                        task = std::move(this->_tasks.front());
                        this->_tasks.pop();
                    }
                    task();
                }
            });
        }
    }

    template<class F, class... Args>
    auto enqueue(F&& f, Args&&... args) 
        -> std::future<typename std::result_of<F(Args...)>::type> {
        using return_type = typename std::result_of<F(Args...)>::type;

        auto task = std::make_shared<std::packaged_task<return_type()>>(
            std::bind(std::forward<F>(f), std::forward<Args>(args)...)
        );
        
        std::future<return_type> res = task->get_future();
        {
            std::unique_lock<std::mutex> lock(_queue_mutex);
            if(_stop)
                throw std::runtime_error("enqueue on stopped ThreadPool");
            _tasks.emplace([task](){ (*task)(); });
        }
        _condition.notify_one();
        return res;
    }

    ~ThreadPool() {
        {
            std::unique_lock<std::mutex> lock(_queue_mutex);
            _stop = true;
        }
        _condition.notify_all();
        for(std::thread &worker: _workers)
            worker.join();
    }

private:
    std::vector<std::thread> _workers;
    std::queue<std::function<void()>> _tasks;
    
    std::mutex _queue_mutex;
    std::condition_variable _condition;
    bool _stop;
};

int main() {
    // 创建一个包含4个工作线程的线程池
    ThreadPool pool(4);

    // 向线程池提交任务
    auto result = pool.enqueue([](int answer) { return answer; }, 42);
    std::cout << "结果: " << result.get() << std::endl;

    // 等待所有任务完成
    std::this_thread::sleep_for(std::chrono::seconds(10));

    return 0;
}