#!/bin/sh
# Copyright 2022-2023 AzukiAtsui

# To use this script, set environment variable AVB_FLAG to [0-3].
# AVB_FLAG=0  # enable AVB;
# AVB_FLAG=1  # disable dm-verity (disable-verity);
# AVB_FLAG=2  # disable boot verification (disable-verification);
# AVB_FLAG=3  # disable both (same effect to
# `fastboot --slot all --disable-verity --disable-verification flash vbmeta vbmeta.img`).
# Example: `AVB_FLAG=3 . avb-switch.sh`
#
# For more info of AVB (Android Verified Boot),
# see <https://source.android.com/docs/security/features/verifiedboot/avb>.

toupper() {
  echo "$@" | tr '[:lower:]' '[:upper:]'
}

grep_cmdline() {
  local REGEX="s/^$1=//p"
  { echo "$(cat /proc/cmdline)$(sed -e 's/[^"]//g' -e 's/""//g' /proc/cmdline)" | xargs -n 1; \
    sed -e 's/ = /=/g' -e 's/, /,/g' -e 's/"//g' /proc/bootconfig; \
  } 2>/dev/null | sed -n "$REGEX"
}

grep_prop() {
  local REGEX="s/^$1=//p"
  shift
  local FILES=$@
  [ -z "$FILES" ] && FILES='/system/build.prop'
  dos2unix $FILES 2>/dev/null | sed -n "$REGEX" | head -n 1
}

# find_block [partname...]
find_block() {
  local BLOCK DEV DEVICE DEVNAME PARTNAME UEVENT
  for BLOCK in "$@"; do
    DEVICE=$(find /dev/block \( -type b -o -type c -o -type l \) -iname "$BLOCK" | head -n 1) 2>/dev/null
    if [ -n "$DEVICE" ]; then
      readlink -f "$DEVICE"
      return 0
    fi
  done
  # Fallback by parsing sysfs uevents
  for UEVENT in /sys/dev/block/*/uevent; do
    DEVNAME=$(grep_prop DEVNAME "$UEVENT")
    PARTNAME=$(grep_prop PARTNAME "$UEVENT")
    for BLOCK in "$@"; do
      if [ "$(toupper "$BLOCK")" = "$(toupper "$PARTNAME")" ]; then
        echo "/dev/block/$DEVNAME"
        return 0
      fi
    done
  done
  # Look just in /dev in case we're dealing with MTD/NAND without /dev/block devices/links
  for DEV in "$@"; do
    DEVICE=$(find /dev \( -type b -o -type c -o -type l \) -maxdepth 1 -iname "$DEV" | head -n 1) 2>/dev/null
    if [ -n "$DEVICE" ]; then
      readlink -f "$DEVICE"
      return 0
    fi
  done
  return 1
}

which_avb_state() {
  AVBTOOL=$(busybox od -w16 -An -tx1 "$VBMETA" |\
    grep -i -B 2 '61 76 62 74 6f 6f 6c 20' |\
    tr -d '[:space:]' |\
    grep -E -oi '0000000000000000000000..00000000617662746f6f6c20')
  echo "- Checking AVB flag state:"
  case $AVBTOOL in
    00000000000000000000000300000000617662746f6f6c20)
      echo "- 03: disabled AVB."
      ;;
    00000000000000000000000200000000617662746f6f6c20)
      echo "- 02: disabled Boot Verification."
      ;;
    00000000000000000000000100000000617662746f6f6c20)
      echo "- 01: disabled Dm-verity."
      ;;
    00000000000000000000000000000000617662746f6f6c20)
      echo "- 00: enabled AVB."
      ;;
    *)
      echo "- Jeez! Field 'avbtool' Not Found."
      echo "- $VBMETA BROKEN or NOT vbmeta???"
      ;;
  esac
  echo ""
}

avb_switch() {
  if [ -z "$AVB_FLAG" ]; then
    echo "- ERROR: variable AVB_FLAG NOT set."
    AVB_FLAG=3
    echo "- Default AVB_FLAG=3"
    echo ""
  fi
  which_avb_state
  echo "- Setting AVB flag to $AVB_FLAG..."
  case $AVB_FLAG in
    3)
      echo "- AVB Disabling..."
      ;;
    2)
      echo "- Boot Verification Disabling..."
      ;;
    1)
      echo "- Dm-verity Disabling..."
      ;;
    0)
      echo "- AVB Enabling..."
      ;;
  esac

  if dd if=imgs/0${AVB_FLAG}.img of="$VBMETA" bs=1 seek=123 conv=notrunc; then
    echo "- Succeed. :-)"
    echo ""
    echo "- Redo check..."
    which_avb_state
  else
    echo "- FAILED!!"
    echo ""
  fi
}

echo "******************"
echo " AVB flag setting "
echo "******************"

# MODDIR is the path where this module files completely installed.
if [ -z "$MODDIR" ]; then
  echo "- avb-switch.sh: set variable MODDIR=${0%/*/*} ."
  MODDIR=${0%/*/*}
fi
if [ -n "$MODPATH" ]; then
  cd "$MODPATH" || abort "- Can NOT cd \`$MODPATH\`."
  echo "- cd \$MODPATH: $MODPATH"
else
  cd "$MODDIR" || abort "- Can NOT cd \`$MODDIR\`."
  echo "- cd \$MODDIR: $MODDIR"
fi

if [ -x /data/adb/ksu/bin/busybox ];then
  echo "- Current busybox binary file is from KernelSU."
elif [ -x /data/adb/magisk/busybox ];then
  export PATH="/data/adb/magisk:$PATH"
  echo "- Current busybox binary file is from Magisk."
fi

# Check A/B slot
SLOT=$(grep_cmdline androidboot.slot_suffix)
if [ -z "$SLOT" ]; then
  SLOT=$(grep_cmdline androidboot.slot)
  [ -z "$SLOT" ] || SLOT=_${SLOT}
fi
[ "$SLOT" = "normal" ] && unset SLOT
[ -z "$SLOT" ] || echo "- Current boot slot: $SLOT ."

VBMETA=$(find_block "vbmeta$SLOT")
echo "- find vbmeta${SLOT}: $VBMETA ."
echo ""

avb_switch
