#!/bin/sh
# Copyright 2023 AzukiAtsui
# <AzukiAtsui@163.com>

# Print module information.
this_module() {
  MODVER=$(grep_prop version "$TMPDIR/module.prop")
  MODVCD=$(grep_prop versionCode "$TMPDIR/module.prop")
  MODDES=$(grep_prop description "$TMPDIR/module.prop")
  print_title "MODULE INFORMATION　"
  ui_print "- Version: $MODVER"
  ui_print "- VersionCode: $MODVCD"
  ui_print "- Description: $MODDES"
  ui_print ""
}

# Figure out which `su` is current device using, Magisk or KernelSU?
which_su() {
  print_title "SU INFORMATION　"
  if [ "$KSU" = "true" ]; then
    ui_print "- Variable KSU is true"
    ui_print "- KernelSU version: $KSU_VER"
    ui_print "- KernelSU version code: $KSU_VER_CODE"
    ui_print "- KernelSU kernel version: $KSU_KERNEL_VER_CODE"
    if [ "$BOOTMODE" != "true" ]; then
      ui_print '- KernelSU module is NOT supported to\\'
      abort "  be installed in custom recovery!!"
    fi
  elif [ "$KSU" = "" ] || [ "$KSU" = "false" ]; then
    if [ -n "$MAGISK_VER" ]; then
      ui_print "- Magisk version: $MAGISK_VER"
      ui_print "- Magisk version code: $MAGISK_VER_CODE"
      MAGISK_SU=true
    else
      abort "- Unknown su!!"
    fi
  fi
  ui_print ""
}

# realpath_which <command> [extra commands...]
realpath_which() {
  local PATHNAME REALPATH_ALL
  # Get executable files pathname from PATH.
  PATHNAME=$(which "$@")
  # Produce a canonicalized absolute pathname.
  REALPATH_ALL=$(realpath "$PATHNAME")
  echo "$REALPATH_ALL" | head -n 1
}

# Setup busybox
which_busybox() {
  print_title "BUSYBOX INFORMATION　"
  BUSYBOX_PATH=$(realpath_which busybox)
  ui_print "- 当前使用的 busybox 是 \`$BUSYBOX_PATH\`。"
  case $BUSYBOX_PATH in
    /data/adb/ksu/bin/busybox)
      # KernelSU ksud added /data/adb/ksu/bin to PATH.
      ui_print "- Current busybox binary file is from KernelSU."
      ;;
    *)
      if [ -x /data/adb/magisk/busybox ]; then
        ui_print "- Found executable /data/adb/magisk/busybox ."
        ui_print "- Add /data/adb/magisk to PATH."
        export PATH="/data/adb/magisk:$PATH"
        ui_print "- Current busybox binary file is from Magisk."
      else
        ui_print "- Unknown busybox path. Quit installation!!"
        abort "- Bad busybox environment!!"
      fi
      ;;
  esac
  ui_print ""
}

clone_perm() {
  local FILE TARGET PERM PERM_NUMBER OWNER GROUP CON
  FILE=$1
  TARGET=$2
  eval "$(ls -l "$FILE" | awk -F " " '{printf("PERM=%s; OWNER=%s; GROUP=%s",$1,$3,$4)}')"
  PERM_NUMBER=$(echo "$PERM" | sed 's/^-//;s/\.//;y/rwx-/4210/;s/.\{1\}/& /g' | \
    awk '{print $1+$2+$3""$4+$5+$6""$7+$8+$9}')
  CON=$(ls -Z "$FILE" | awk -F " " '{print $1}')
  set_perm "$TARGET" "$OWNER" "$GROUP" "$PERM_NUMBER" "$CON"
}

clone_perm_recursive() {
  true
}

# Ensure file in /data/ writable.
chattr_minus_i() {
  chattr -i "$1" && ui_print "- Unlocked(chattr -i) $1 ."
}

# Lock the specified file or directory in order to
# prevent it from being deleted or modified.
chattr_plus_i() {
  chattr +i "$1" && ui_print "- Locked(chattr +i) $1 ."
}

# f_rec_data_file <original filename> <COPYFILE>
f_rec_data_file() {
  {
    echo "chattr_minus_i \"$1\""
    echo "cp \"$2\" \"$1\""
    echo "clone_perm \"$2\" \"$1\""
    #echo "chattr_plus_i \"$1\""
  } >> "$MOD_ROOT/uninstall.sh"
}

symlink_path() {
  # Auto Generated, DO NOT MANUALLY CREATE OR MODIFY
  # <https://kernelsu.org/guide/module.html#kernelsu-modules>
  # <https://topjohnwu.github.io/Magisk/guides.html#magisk-modules>
  realpath "$1" | sed 's|^/vendor/|/system&|;s|^/product/|/system&|;s|^/system_ext/|/system&|'
}

check_copy() {
  # Check if file exists.
  if [ ! -e "$1" ]; then
    ui_print "- ERROR: NOT existing $1 !"
    return 1
  fi
  # Transform file path.
  COPYFILE_TAIL=$(symlink_path "$1")
  COPYFILE_HEAD=$MOD_ROOT
  # Backup /data/ files to AZUKI_BACKUP_DIR.
  [ -n "$(echo "$COPYFILE_TAIL" | sed -n '/^\/data\//p')" ] && COPYFILE_HEAD=$AZUKI_BACKUP_DIR
  COPYFILE=$COPYFILE_HEAD$COPYFILE_TAIL
  #COPYFILE_FINAL=$MODDIR$COPYFILE_TAIL
  check_copyfile(){
    # Going to modify COPYFILE unless original files in /data.
    if [ -n "$(echo "$COPYFILE_TAIL" | sed -n '/^\/data\//p')" ]; then
      IS_IN_DATA=true
      chattr_minus_i "$1"
      # Recovery backup while uninstall module.
      f_rec_data_file "$COPYFILE_TAIL" "$COPYFILE"
    else
      IS_IN_DATA=false
    fi
  }
  # Create file if not exists.
  if [ ! -e "$COPYFILE" ]; then
    mktouch "$COPYFILE"
    cp "$1" "$COPYFILE"
    clone_perm "$1" "$COPYFILE"
    ui_print "- Cloned $1 to $COPYFILE ."
    check_copyfile "$1"
  else
    ui_print "- Existing $COPYFILE ."
    check_copyfile "$1"
  fi
}

# which_mount <mount target file> [modified file pathname: default $MODDIR$1]
which_mount() {
  if [ -z "$2" ]; then
    MOUNT_FILE=$MODDIR$1
  else
    MOUNT_FILE=$2
  fi
  if [ "$KSU" = "true" ]; then
    # Reference: <https://wiki.archlinux.org/title/Overlay_filesystem>
    # OverlayFS works on directories.
    MOUNT_TARGET_ROOT=$(echo "$1" | sed -n 's|\(^/[a-zA-Z_]*\)/.*|\1|;p')
    #MOUNT_TARGET_DIR=${1%/*}
    MOUNT_UPPERDIR=$MODDIR$MOUNT_TARGET_ROOT
    ui_print "- mount -t overlay -o lowerdir=$MOUNT_UPPERDIR:$MNT$MOUNT_TARGET_ROOT overlay $MNT$MOUNT_TARGET_ROOT >> $P_F_D"
    echo "mount -t overlay -o lowerdir=$MOUNT_UPPERDIR:$MNT$MOUNT_TARGET_ROOT overlay $MNT$MOUNT_TARGET_ROOT" >> "$P_F_D"
    return 0
  fi
  if [ "$MAGISK_VER_CODE" -lt 25200 ]; then
    ui_print  "- mount --bind $MOUNT_FILE $1 >> $P_F_D"
    echo "mount --bind $MOUNT_FILE $1" >> "$P_F_D"
  else
    ui_print  "- mount -o bind $MOUNT_FILE $1 >> $P_F_D"
    echo "mount -o bind $MOUNT_FILE $1" >> "$P_F_D"
  fi
}

# file_mod_template <original file> <shells function>
file_mod_template() {
  if check_copy "$1"; then
    if $IS_IN_DATA; then
      $2 "$1"
      del_extra "$1"
      # Prevent it from modification from cloud.
      chattr_plus_i "$1"
    else
      # Read-only files system-less modification.
      $2 "$COPYFILE"
      del_extra "$COPYFILE"
      if [ "$(echo "$COPYFILE_TAIL" | sed -n '/^\/system\//p')" = '' ]; then
        which_mount "$COPYFILE_TAIL"
      else
        ui_print "- COPYFILE in MODPATH/system/ does not need to add mount command line to post-fs-data.sh"
      fi
    fi
    return 0
  else
    ui_print "- 发生错误，忽略修改！！"
    return 1
  fi
}

package_list_3() {
  pm list packages -3 | sed 's/^package://'
}

battery_background() {
  for item in $(package_list_3); do
    # Android Doze whitelist(电池优化白名单)
    dumpsys deviceidle whitelist +${item}
    # To simulate conditions where implicit broadcasts and background services are unavailable, enter the following command:
    # adb shell cmd appops set <package_name> RUN_IN_BACKGROUND ignore
    # To re-enable implicit broadcasts and background services, enter the following command:
    # adb shell cmd appops set <package_name> RUN_IN_BACKGROUND allow
    # 允许应用程序在后台运行
    cmd appops set $item RUN_IN_BACKGROUND allow
    cmd appops set $item RUN_ANY_IN_BACKGROUND allow
  done
}

which_api() {
  ui_print "- API level: $API"
  # What is API Level? See <https://developer.android.com/guide/topics/manifest/uses-sdk-element#ApiLevels>.
  if [ "$API" -lt 30 ]; then
    abort "- Not support Android 10 and older Android version."
  elif [ "$API" -eq 30 ]; then
    ui_print "- Hello, Android 11 user. •ᴗ•"
  elif [ "$API" -le 32 ]; then
    ui_print "- Hello, Android 12 user. ❛‿˂̵✧"
  elif [ "$API" -eq 33 ]; then
    ui_print "- Hello, Android 13 user. (＾Ｕ＾)ノ~"
  fi
}

# Get properties of the devices produced by Oplus.
oplus_system_prop() {
  NVID=$(getprop ro.build.oplus_nv_id)
  if [ -z "$NVID" ]; then
    return 1
  fi
  case $NVID in
    10010111) OPLUS_REGION="CN 中国 China";;
    00011010) OPLUS_REGION="TW 中国台湾省 Taiwan";;
    00110111) OPLUS_REGION="RU 俄罗斯 Russia";;
    01000100) OPLUS_REGION="GDPR 欧盟 EU";;
    10001101) OPLUS_REGION="GDPR 欧洲 Europe";;
    00011011) OPLUS_REGION="IN 印度 India";;
    00110011) OPLUS_REGION="ID 印度尼西亚 Indonesia";;
    00111000) OPLUS_REGION="MY 马来西亚 Malaysia";;
    00111001) OPLUS_REGION="TH 泰国 Thailand";;
    00111110) OPLUS_REGION="PH 菲律宾 Philippines";;
    10000011) OPLUS_REGION="SA 沙特阿拉伯 Saudi Arabia";;
    10011010) OPLUS_REGION="LATAM 拉丁美洲 Latin America";;
    10011110) OPLUS_REGION="BR 巴西 Brazil";;
    10100110) OPLUS_REGION="MEA 中东和非洲 The Middle East and Africa";;
    *) OPLUS_REGION="unknown carrier identifier: $NVID";;
  esac
  ui_print "- Market Name: $(getprop ro.vendor.oplus.market.name)"
  PRODUCT_ID=$(getprop ro.commonsoft.ota)
  ui_print "- Device ID: $PRODUCT_ID"
  PROJECT_NAME=$(getprop ro.separate.soft)  # ro.boot.prjname
  ui_print "- Project Name: $PROJECT_NAME"
  # Realme realmeUI
  RUI_VER=$(getprop ro.build.version.realmeui)
  [ -z "$RUI_VER" ] || ui_print "- realmeUI Version: $RUI_VER"
  # OPPO(includes OnePlus) ColorOS
  COS_VER=$(getprop ro.build.version.oplusrom)
  ui_print "- ColorOS Version: $COS_VER"
  ui_print "- Current device may be purchased in \"$OPLUS_REGION\"."
  ui_print ""
  return 0
}

which_system() {
  print_title "DEVICE INFORMATION　"
  PRODUCT_BRAND=$(getprop ro.product.vendor.brand)
  ui_print "- Brand: $PRODUCT_BRAND"
  PRODUCT_NAME=$(getprop ro.product.name)
  ui_print "- Product: $PRODUCT_NAME"
  PRODUCT_MODEL=$(getprop ro.product.model)
  ui_print "- Model: $PRODUCT_MODEL"
  SOC_MODEL=$(getprop ro.soc.model)
  ui_print "- SOC: $SOC_MODEL"
  CODENAME=$(sed -n 's/.*[A-Z0-9][ ,]//;p' /sys/firmware/devicetree/base/model)
  ui_print "- Codename: $CODENAME"
  ui_print "- CPU architecture: $ARCH"
  which_api
  print_title "$(toupper "$PRODUCT_BRAND") SYSTEM PROPERTIES　"
  if oplus_system_prop; then
    SYSTEMTYPE=oplus
    return 0
  fi
  abort "- Not support current system."
}

flash_image() {
  local CMD1
  case "$1" in
    *.gz) CMD1="gzip -d < '$1' 2>/dev/null";;
    *)    CMD1="cat '$1'";;
  esac
  if [ -b "$2" ]; then
    local img_sz=$(stat -c '%s' "$1")
    local blk_sz=$(blockdev --getsize64 "$2")
    [ "$img_sz" -gt "$blk_sz" ] && return 1
    blockdev --setrw "$2"
    local blk_ro=$(blockdev --getro "$2")
    [ "$blk_ro" -eq 1 ] && return 2
    eval "$CMD1" | cat - /dev/zero > "$2" 2>/dev/null
  elif [ -c "$2" ]; then
    flash_eraseall "$2" >&2
    eval "$CMD1" | nandwrite -p "$2" - >&2
  else
    ui_print "- Not block or char device, storing image"
    eval "$CMD1" > "$2" 2>/dev/null
  fi
  return 0
}

mark_remove_print() {
  mark_remove "$MOD_ROOT$(symlink_path "$1")"
  ui_print "- KernelSU REMOVE: $1"
  if [ -n "$2" ]; then
    shift
    mark_remove_print "$@"
  fi
}

mark_replace_print() {
  mark_replace "$MOD_ROOT$(symlink_path "$1")"
  ui_print "- KernelSU REPLACE: $1"
  if [ -n "$2" ]; then
    shift
    mark_replace_print "$@"
  fi
}

mktouch_replace() {
  mktouch "$MOD_ROOT$(symlink_path "$1")/.replace"
  ui_print "- Magisk REPLACE: $1"
  if [ -z "$(echo "$1" | sed -n '/^\/vendor\//p;/^\/product\//p;/^\/system_ext\//p')" ]; then
    ui_print "- Magisk REPLACE only works for MODPATH/system directory by default."
    which_mount "$1"
  fi
  if [ -n "$2" ]; then
    shift
    mktouch_replace "$@"
  fi
}

avb() {
  AVB_FLAG=$1 . "$MOD_ROOT/scripts/avb-switch.sh"
}

fix_local_execute() {
  # 补齐函数; 设置变量
  . "$MOD_ROOT/scripts/util-functions.sh"
}

main_env() {
  which_su
  which_busybox
  which_system
}

# MODDIR is the path where this module files completely installed.
if [ -z "$MODDIR" ]; then
  ui_print "- check-env.sh: set variable MODDIR=${0%/*/*} ."
  MODDIR=${0%/*/*}
fi
if [ "$INSTALLING" = "true" ]; then
  MOD_ROOT=$MODPATH
  this_module
else
  MOD_ROOT=$MODDIR
  fix_local_execute
fi

# Import edit functions.
. "$MOD_ROOT/scripts/edit-functions.sh"

XMLHEAD="<?xml version='1.0' encoding='UTF-8' standalone='yes' ?>"
AZUKI_BACKUP_DIR=/data/adb/AzukiAtsui
P_F_D="$MOD_ROOT/post-fs-data.sh"
# Mount point.
MNT=/mnt/vendor

main_env
